/*
 * All cairo-related code may be found in the files starting with cairo-*.
 */

#include "EGL/egl.h"
#include "GLES2/gl2.h"
#include "GLES2/gl2ext.h"

#include <wayland-client.h>
#include <wayland-egl.h>
#include "wayland-util.h"
#include "compositor-shim.h"
#include <wayland_backend.h>
#include "cairo-drawing.h"

#define SCREEN_WIDTH 1024
#define SCREEN_HEIGHT 768
#define RUNMODE_COUNT 10

static void swap_buffer();
static void egl_open();
static void egl_close();
static void buffer_init(void);
static void render_cairo_wl(struct modeset_dev *);
static void terminate_renderer(void);
static void handle_egl_error(const char *);
static void init_renderer(uint8_t *buffer, uint16_t width, uint16_t height);

typedef struct _WaylandGles WaylandGles;

struct _WaylandGles
{
	struct wl_display*    wl_display;
	struct wl_surface*    wl_surface;
	struct wl_registry*   wl_registry;
	struct wl_egl_window* wl_native_window;
	struct wl_compositor* wl_compositor;
	unsigned int          surface_id;
	unsigned int          layer_id;

	struct compositor_shim_context*        wlAdapterContext;
	struct compositor_shim_surface_context wlAdapterSurfaceContext;
};

typedef struct egl_setup_t
{
	EGLContext egl_context;
	EGLSurface egl_surface;
	EGLDisplay egl_display;
	WaylandGles wland_info;
	EGLConfig    eglConfig;
}egl_setup;

egl_setup egl_object;
egl_setup egl_object_shared;

static WaylandGles aWayland;

int wayland_init()
{
	egl_open();
	buffer_init();
	egl_close();
	return 0;
}

/**
 * @brief modeset_draw
 * @param[in]  void
 * @param[out] void
 * modeset_draw() allocates memory to the buffer and sets its value .
 * The rendering has moved into another helper function modeset_draw_dev().
 * It is also responsible to free the allocated memory.
 */
static void buffer_init()
{
	struct modeset_buf *buf;
	struct modeset_dev *dev;
	int runmode = RUNMODE_COUNT;

	dev = malloc(sizeof( struct modeset_dev));
		memset(dev, 0, sizeof(struct modeset_dev));
	buf = &dev->bufs[dev->front_buf];
	buf->map = (uint8_t *)malloc(SCREEN_WIDTH * SCREEN_HEIGHT * 4);
	buf->width = SCREEN_WIDTH;
	buf->height = SCREEN_HEIGHT;
	buf->stride = SCREEN_WIDTH*4;

	while ( runmode )
	{
		render_cairo_wl(dev);
		runmode--;
	}

	free(buf->map);
	free(dev);
	terminate_renderer();
}

/**
 * @brief modeset_draw_dev
 * modeset_draw_dev() is a new function that redraws the screen of a single
 * output.
 * @param[in]  pointer to structure modeset_dev
 * @param[out] void
 * It redraws a new frame and performs swap.
 */
static void render_cairo_wl(struct modeset_dev *dev)
{
	modeset_draw_cairo(dev->bufs[dev->front_buf].map,
				dev->bufs[dev->front_buf].width,
				dev->bufs[dev->front_buf].height,
				dev-> bufs[dev->front_buf].stride);

	init_renderer(dev->bufs[dev->front_buf].map,
			dev->bufs[dev->front_buf].width,
			dev->bufs[dev->front_buf].height);

	next_state();
	swap_buffer();
}

void RegistryHandleGlobal(void* data, struct wl_registry* registry,
			uint32_t name, const char* interface, uint32_t version)
{
	version = version;
	WaylandGles * sink = (WaylandGles *) data;
	if (!strcmp(interface, "wl_compositor"))
	{
		sink->wl_compositor = (struct wl_compositor*)(wl_registry_bind(registry,
					name, &wl_compositor_interface, 1));
	}
}

static struct wl_registry_listener registryListener = {
		RegistryHandleGlobal,
		NULL
};

static EGLBoolean create_wl_context(WaylandGles * sink)
{
		sink->wl_display = wl_display_connect(NULL);
		if (sink->wl_display == NULL)
		{
			printf("fail wl_display_connect\n");
			return EGL_FALSE;
		}

		sink->wl_registry = wl_display_get_registry(sink->wl_display);

		wl_registry_add_listener(sink->wl_registry,&registryListener, sink);

		wl_display_dispatch(sink->wl_display);
		wl_display_roundtrip(sink->wl_display);

		if(sink->wl_compositor == NULL)
		{
			printf("fail wl_compositor\n");
			return EGL_FALSE;
		}

		sink->wlAdapterContext = compositor_shim_initialize (sink->wl_display);
		if (NULL == sink->wlAdapterContext)
		{
			printf("fail wl compositor_shim_initialize\n");
			return EGL_FALSE;
		}

		sink->wl_surface = wl_compositor_create_surface(sink->wl_compositor);
		if(sink->wl_surface == NULL)
		{
			printf("fail wl_compositor_create_surface 4\n");
			return EGL_FALSE;
		}

		sink->wl_native_window = wl_egl_window_create(sink->wl_surface,
						SCREEN_WIDTH, SCREEN_HEIGHT);
		if (sink->wl_native_window == NULL)
		{
			printf("fail wl_egl_window_create 5\n");
			return EGL_FALSE;
		}

	return EGL_TRUE;
}

/**
 * @brief egl_open
 * @param[in]  void
 * @param[out] void
 * sets up an on-screen EGL rendering surface
 */
static void egl_open()
{
	EGLBoolean ret;

	static const EGLint gl_context_attribs[] =
	{
		EGL_CONTEXT_CLIENT_VERSION, 2,
		EGL_NONE
	};

	EGLint s_configAttribs[] =
	{
		EGL_RED_SIZE,			8,
		EGL_GREEN_SIZE,			8,
		EGL_BLUE_SIZE,			8,
		EGL_ALPHA_SIZE,			8,
		EGL_DEPTH_SIZE,			EGL_DONT_CARE,
		EGL_STENCIL_SIZE,		EGL_DONT_CARE,
		EGL_SURFACE_TYPE,		EGL_WINDOW_BIT,
		EGL_RENDERABLE_TYPE,	EGL_OPENGL_ES2_BIT,
		EGL_SAMPLES,			EGL_DONT_CARE,
		EGL_NONE
	};
	EGLint numConfigs;
	EGLint majorVersion;
	EGLint minorVersion;

	/* set wayland environment to run without shell script */
	if(NULL == getenv("XDG_RUNTIME_DIR"))
	{
		setenv("XDG_RUNTIME_DIR","/tmp/",EGL_TRUE);
	}

	WaylandGles* sink = &aWayland;
	int adapter_error = 0;
	sink->surface_id = 40;
	sink->layer_id = 3000;
	if (!create_wl_context(sink))
	{
		printf("fail 2\n");
		return ;
	}

	egl_object.egl_display = eglGetDisplay( (EGLNativeDisplayType)sink->wl_display);

	adapter_error = compositor_shim_surface_init (&sink->wlAdapterSurfaceContext,
	sink->wl_surface,sink->layer_id,sink->surface_id, SCREEN_WIDTH,SCREEN_HEIGHT);
	if (adapter_error != 0)
	{
		printf("compositor_shim_surface_init() failed\n");
		return;
	}

	adapter_error = compositor_shim_surface_configure (sink->wlAdapterContext,
	&sink->wlAdapterSurfaceContext,ADAPTER_CONFIGURATION_ALL);
	if (adapter_error != 0)
	{
		printf("compositor_shim_surface_configure() failed\n");
		return;
	}

	s_configAttribs[16] =  EGL_NONE;
	eglBindAPI(EGL_OPENGL_ES_API);

	eglInitialize(egl_object.egl_display, &majorVersion, &minorVersion);

	eglGetConfigs(egl_object.egl_display, NULL, 0, &numConfigs);

	eglChooseConfig(egl_object.egl_display, s_configAttribs,
			& egl_object.eglConfig, 1, &numConfigs);

	egl_object.egl_surface = eglCreateWindowSurface(egl_object.egl_display,
		egl_object.eglConfig,(EGLNativeWindowType)sink->wl_native_window,NULL);
	if (egl_object.egl_surface == EGL_NO_SURFACE)
		handle_egl_error ("eglCreateWindowSurface");

	egl_object.egl_context = eglCreateContext(egl_object.egl_display,
			egl_object.eglConfig, EGL_NO_CONTEXT,gl_context_attribs);
	if (egl_object.egl_context == EGL_NO_CONTEXT)
		handle_egl_error("eglCreateContext");

	ret = eglMakeCurrent(egl_object.egl_display,egl_object.egl_surface,
				egl_object.egl_surface,egl_object.egl_context);
	if (ret == EGL_FALSE)
		handle_egl_error("eglMakeCurrent");
	else
		{
			printf("setup ok\n" );
		}
		eglSwapInterval(egl_object.egl_display,1);
}

static void handle_egl_error(const char *name)
{
	static char * const error_strings[] =
	{
		"EGL_SUCCESS",
		"EGL_NOT_INITIALIZED",
		"EGL_BAD_ACCESS",
		"EGL_BAD_ALLOC",
		"EGL_BAD_ATTRIBUTE",
		"EGL_BAD_CONFIG",
		"EGL_BAD_CONTEXT",
		"EGL_BAD_CURRENT_SURFACE",
		"EGL_BAD_DISPLAY",
		"EGL_BAD_MATCH",
		"EGL_BAD_NATIVE_PIXMAP",
		"EGL_BAD_NATIVE_WINDOW",
		"EGL_BAD_PARAMETER",
		"EGL_BAD_SURFACE"
	};
	EGLint error_code=eglGetError();

	fprintf(stderr," %s: egl error \"%s\" (0x%x)\n", name,
			error_strings[error_code-EGL_SUCCESS], error_code);
	exit(EXIT_FAILURE);
}

/**
 * @brief egl_close
 * @param[in]  void
 * @param[out] void
 * destroys on-screen EGL rendering surface
 */
static void egl_close(void)
{
	eglMakeCurrent(egl_object.egl_display, NULL, NULL, NULL);
	eglDestroyContext(egl_object.egl_display, egl_object.egl_context);
	eglDestroySurface(egl_object.egl_display,egl_object.egl_surface);
	eglTerminate(egl_object.egl_display);

	WaylandGles* sink = &aWayland;
	wl_display_flush( sink->wl_display );
	wl_display_roundtrip( sink->wl_display);
	compositor_shim_terminate(sink->wlAdapterContext);

	if (sink->wl_native_window)
	{
		wl_egl_window_destroy(sink->wl_native_window);
	}

	if (sink->wl_surface)
	{
		wl_surface_destroy(sink->wl_surface);
	}

	if (sink->wl_compositor)
	{
		wl_compositor_destroy(sink->wl_compositor);
	}
	if (sink->wl_display)
	{
		wl_display_disconnect( sink->wl_display);
	}
}

/**
 * @brief swap_buffer
 *
 *@param[in] void
 *@param[out] void
 * swap the buffers
 */
static void swap_buffer()
{
	eglSwapBuffers(egl_object.egl_display, egl_object.egl_surface);
	sleep(4);
}

typedef struct _es2shaderobject
{
	GLuint fragmentShaderId;
	GLuint vertexShaderId;
	GLuint shaderProgramId;
} ES2ShaderObject;

typedef struct _es2vertexbufferobject
{
	GLuint vbo;
} ES2VertexBuffer;

static ES2ShaderObject	gShaderObject;
static ES2VertexBuffer	gVertexBuffer;
static 	GLuint uTexToRender;

// Vertex shader code
char* pszVertShaTexMap = "\
		attribute highp vec4    vertPos;\
		uniform mediump mat4    MVMatrix;\
		attribute highp vec2    tex_coord;\
		varying highp vec2      tex_lookup;\
        void main(void)\
		{\
			tex_lookup = tex_coord;\
			gl_Position = vertPos * MVMatrix;\
		}";

// Fragment shader code
char* pszFragShaTexMap = "\
		uniform mediump sampler2D    texobj;\
		varying highp vec2   tex_lookup;\
		precision mediump float;\
		void main(void)\
		{\
		vec2 flipped_texcoord = vec2(tex_lookup.x, 1.0 - tex_lookup.y);\
		gl_FragColor = texture2D(texobj, flipped_texcoord);\
        }";


bool InitShader()
{
	GLint result = 0;
	GLuint i;
	char pszInfoLog[1024];
	int  nInfoLogLength;

	// Create the vertex shader object
	gShaderObject.vertexShaderId = glCreateShader(GL_VERTEX_SHADER);
	glShaderSource(gShaderObject.vertexShaderId, 1,
			(const char**)&pszVertShaTexMap, NULL);
	glCompileShader(gShaderObject.vertexShaderId);

	glGetShaderiv(gShaderObject.vertexShaderId, GL_COMPILE_STATUS, &result);
	if (!result)
	{
		printf("Error: Failed to compile GLSL shader #%d\n",i);
		glGetShaderInfoLog(gShaderObject.vertexShaderId, 1024, &nInfoLogLength, pszInfoLog);
		printf("%s",pszInfoLog);
		return false;
	}
	// Create the fragment shader object
	gShaderObject.fragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
	glShaderSource(gShaderObject.fragmentShaderId, 1,
			(const char**)&pszFragShaTexMap, NULL);
	glCompileShader(gShaderObject.fragmentShaderId);

	glGetShaderiv(gShaderObject.fragmentShaderId, GL_COMPILE_STATUS, &result);
	if (!result)
	{
		printf("Error: Failed to compile GLSL shader #%d\n",i);
		glGetShaderInfoLog(gShaderObject.fragmentShaderId, 1024, &nInfoLogLength, pszInfoLog);
		printf("%s",pszInfoLog);
		return false;
	}

	gShaderObject.shaderProgramId = glCreateProgram();
	glAttachShader(gShaderObject.shaderProgramId,
			gShaderObject.fragmentShaderId);
	glAttachShader(gShaderObject.shaderProgramId,
			gShaderObject.vertexShaderId);
	glLinkProgram(gShaderObject.shaderProgramId);

	glGetProgramiv(gShaderObject.shaderProgramId, GL_LINK_STATUS, &result);
	if (!result)
	{
		printf("Error: Failed to validate GLSL program\n");
		glGetProgramInfoLog(gShaderObject.shaderProgramId, 1024, &nInfoLogLength, pszInfoLog);
		printf("%s",pszInfoLog);
		return false;
	}

	glUseProgram(gShaderObject.shaderProgramId);
	return true;
}

static void init_renderer(uint8_t *pTex,uint16_t width,uint16_t height)
{
	glViewport(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
	GLint tmapvPos;
	GLint tmapTexPos;
	GLint MVmapPos;
	GLint texsmapler;
	GLfloat vertsForTexMap[] =
			{-1.0f, -1.0f, 0.0f,
			1.0f, -1.0f, 0.0f,
			-1.0f,  1.0f, 0.0f,
			1.0f,  1.0f, 0.0f,
			0.0f,  0.0f, 0.0f,
			1.0f,  0.0f, 0.0f,
			0.0f,  1.0f, 1.0f,
			1.0f,  1.0f, 0.0f};

	float pfIdentity[4][4] =
			{{1.0f,0.0f,0.0f,0.0f},
			{0.0f,1.0f,0.0f,0.0f},
			{0.0f,0.0f,1.0f,0.0f},
			{0.0f,0.0f,0.0f,1.0f}};

	if (!InitShader())
	{
		printf("Error: Failed InitShader\n");
	}

	eglSwapInterval(egl_object.egl_display, 1);
	glGenTextures(1, &uTexToRender);
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, uTexToRender);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_BGRA_EXT, 1024,768, 0, GL_BGRA_EXT,
			GL_UNSIGNED_BYTE, pTex);

	glGenBuffers(1, &gVertexBuffer.vbo);
	glBindBuffer(GL_ARRAY_BUFFER, gVertexBuffer.vbo);
	glBufferData(GL_ARRAY_BUFFER, sizeof(vertsForTexMap), vertsForTexMap,
			GL_STATIC_DRAW);

	glClearColor(0.0, 0.0, 0.0, 1.0);
	glClear(GL_COLOR_BUFFER_BIT);

	tmapvPos = glGetAttribLocation(gShaderObject.shaderProgramId, "vertPos");
	glEnableVertexAttribArray(tmapvPos);
	glVertexAttribPointer(tmapvPos, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)0);

	tmapTexPos = glGetAttribLocation(gShaderObject.shaderProgramId, "tex_coord");
	glEnableVertexAttribArray(tmapTexPos);
	glVertexAttribPointer(tmapTexPos, 3, GL_FLOAT, GL_FALSE, 0,
				(GLvoid *)((sizeof(GLfloat)) * 12));

	MVmapPos = glGetUniformLocation(gShaderObject.shaderProgramId, "MVMatrix");
	glUniformMatrix4fv(MVmapPos, 1, GL_FALSE, (GLfloat *)pfIdentity);

	texsmapler = glGetUniformLocation(gShaderObject.shaderProgramId, "texobj");
	glUniform1i(texsmapler, 0);

	glDrawArrays(GL_TRIANGLE_STRIP,0,4);
	glFinish();
}

static void terminate_renderer()
{
	glDeleteProgram(gShaderObject.shaderProgramId);
	glDeleteShader(gShaderObject.fragmentShaderId);
	glDeleteShader(gShaderObject.vertexShaderId);
	glDeleteBuffers(1, &gVertexBuffer.vbo);
	glDeleteTextures(1, &uTexToRender);
}
